% ShoreShop 2.0 CoSMoS-COAST-CONV (convolution) model by Sean Vitousek.
% Edited by Jakob C. Christiaanse for sea turle beach modeling.
clear all; close all; clc;

% If the input data is not in the same directory as this file, add the path to the data
% cwd = 'YOUR/PATH/';   % add your path
cwd = '';               % leave empty if the data is in the same directory as this file

loc = 'AI_LB';          % specify the location identifier
slr_scen = 'hind'       % specify SLR scenario ('hind' for hindcast)

% specify run ID and output path
% you might need to adjust this to your directory structure
if strcmp(slr_scen, 'hind')
    run_id = slr_scen;
    output_dir = [cwd, loc, '/1_hindcast/'];
else
    run_id = ['slr' slr_scen];
    output_dir = [cwd, loc, '/2_forecast/'];
end

% Create output directory if it does not exist
if ~exist(output_dir, 'dir')
    mkdir(output_dir);
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Load forcing conditions
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

fprintf('loading wave/sea-level data ... '); % display progress

% load historical wave conditions
T=readtable([cwd, loc, '/0_input/', loc, '_waves_1979_2024.csv'],'Delimiter',',');     % load the wave height file to a matlab table
t_w=datenum(T.Time); % get the time vector
Hs=T.Hs;             % get Hs
Tp=T.Tp;             % get Tp
Dir=T.Dir;           % get Dir

% For future projections, we load projected waves and concat them to the hindcast
if ~strcmp(slr_scen, 'hind')
    T=readtable([cwd, loc, '/0_input/', loc, '_waves_2025_2100.csv'],'Delimiter',',');     % load the wave height file to a matlab table
    t_w=cat(1,t_w,datenum(T.Time)); % get the time vector
    Hs=cat(1,Hs,T.Hs);              % get Hs
    Tp=cat(1,Tp,T.Tp);              % get Tp
    Dir=cat(1,Dir,T.Dir);           % get Dir
end

% smooth waves (this is mostly done to remove the few odd NaN's that are present)
smooth_fac=0.001;                         % smoothing factor parameter
Hs_smooth=smoothn(Hs,smooth_fac); % also smooth the alongshore median wave height
Tp_smooth=smoothn(Tp,smooth_fac); % also smooth the alongshore median wave period

Dir0=nanmedian(Dir);                            % calculate the median wave direction (one scalar value)

Dir_rel=Dir-Dir0;                                % recenter the direction data by removing the median
Dir_rel(Dir_rel>180)=Dir_rel(Dir_rel>180)-360;   % recenter values greater than 180 degrees

Dir_rel(abs(Dir_rel)>45)=NaN;                    % get rid of highly oblique waves

Dir_smooth=smoothn(Dir_rel,smooth_fac); % smooth the direction

Dir_smooth=Dir_smooth-nanmedian(Dir_smooth);  % make sure the Dir_med has a zero median value

% check the wave direction
% figure; 
% subplot(3,1,1); plot(t_w,Hs_smooth,'b'); datetick('x')
% subplot(3,1,2); plot(t_w,Tp_smooth,'b'); datetick('x')
% subplot(3,1,3); plot(t_w,Dir_smooth,'b'); datetick('x')

% load the sea-level data
T=readtable([cwd, loc, '/0_input/', loc, '_sealevel_1979_2024.csv'],'Delimiter',',');     
t_sl=datenum(string(T.Time),'YYYY');                                        % get the time vector
SL=T.wl;

% if forecast, load projected sea-level conditions and concat them to the hindcasted ones
if ~strcmp(slr_scen, 'hind')
    T=readtable([cwd, loc, '/0_input/', loc, '_sealevel_2025_2100.csv'],'Delimiter',',');
    t_sl=cat(1,t_sl,datenum(string(T.Time),'YYYY')); % get the time vector
    slr = table2array(T(:, ['wl_' slr_scen]));
    SL=cat(1,SL,slr);
end

SL_smooth=smoothn(SL,1);                                                % smooth the sea-level curve
SL_w=interp1(t_sl,SL_smooth,t_w,'linear','extrap');             % interpolate SL onto the wave times
SL_w(t_w<t_sl(1))=SL_smooth(1);                               % use a flat part of the initial curve

fprintf('done.\n'); % display progress

% figure; plot(t_w,SL_w); datetick('x');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% model time stepping conditions
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

dt=1; % model time step in days 
t0=datenum(1979,1,01);    % model start time
if strcmp(slr_scen, 'hind')
    tstop=datenum(2024,12,31);   % model stop time
else
    tstop=datenum(2100,12,31);   % model stop time
end

% model time vector
t=(t0:dt:tstop)';

% number of time steps
len=length(t);

t_output=t';  % make time array for the short-term prediction

% get forcing onto model time
[Hs]=interp1(t_w,Hs_smooth,t);
[Tp]=interp1(t_w,Tp_smooth,t);
[Dir]=interp1(t_w,Dir_smooth,t);
[SL]=interp1(t_w,SL_w,t,'linear','extrap');

% NOTE: we apply only ONE time series of wave height, period (not used),
% direction, and sea level to ALL transects.  Which is a bit different than
% applying individual/unique wave/SLR time series for each model transect.
% But Beach X is so small that unique time series are unnecessary any
% longshore variability will manifest in the alongshore variability of the
% model parameters... Besides unique wave time series are generally TOO
% FAR offshore to adequately capture the correct (i.e., 'nearshore') nature
% of the wave conditions, anyway.  

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Model parameters and physics
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Load transect-specific slopes
fprintf('loading transect specific beach slopes ... '); % display progress
slopes_file = ['slopes/', loc, '_slopes.csv'];
Tslopes = readtable(slopes_file);
tanAlpha = table2array(Tslopes(1,2:end)); % 1xNtr vector, skip first column
fprintf('done.\n')

% Read fixed parameters from CSV file
param_file = [cwd, loc, '/1_hindcast/', loc, '_hind_parameters.csv'];
Tparams = readtable(param_file);
P = table2array(Tparams(:,2:end)); % rows: parameters, columns: transects
Ntr = size(P,2);

% Model functions
f_cs = @(p,t,Hs) -p(2)/p(1)*(detrend(Hs.^2));
Q = @(p,t,Hs,alpha) p(4).*sind(2*alpha);
f_ls = @(p,t,Hs,alpha) 1./p(3)*detrend(Q(p,t,Hs,alpha),0);
Yvlt = @(p,t) p(6)/365.25*(t-t(1))+p(7);
g_cs = @(p,t) (1-dt/p(1)).^(t/dt).*(t>=0);
g_ls = @(p,t) (1-dt/p(3)).^(t/dt).*(t>=0);
Yst = @(p,t,Hs) cat(1,0,dt*getfield(fconv2(f_cs(p,t,Hs),g_cs(p,t-t(1))),{1:length(t)-1,1}));
Ylst = @(p,t,Hs,alpha) cat(1,0,dt*getfield(fconv2(f_ls(p,t,Hs,alpha),g_ls(p,t-t(1))),{1:length(t)-1,1}));

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Run model for each transect with fixed parameters
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Yfinal = NaN(length(t),Ntr);
YLST = NaN(length(t),Ntr);
YST = NaN(length(t),Ntr);
YBRU = NaN(length(t),Ntr);
YVLT = NaN(length(t),Ntr);

for i=1:Ntr
    p = P(:,i);

    % Define the bruun and full model functions for each transect specific slope
    Ybru = @(p, t, S) -p(5)/tanAlpha(i)*(S-S(1));
    Ymod = @(p, t, Hs, alpha, S) Yst(p, t, Hs) + Ylst(p, t, Hs, alpha) + Ybru(p, t, S) + Yvlt(p, t);
    Yfinal(:,i) = Ymod(p, t, Hs, Dir, SL);
    YLST(:,i) = Ylst(p, t, Hs, Dir);
    YST(:,i) = Yst(p, t, Hs);
    YBRU(:,i) = Ybru(p, t, SL);
    YVLT(:,i) = Yvlt(p, t);
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% make .csv output
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% make/interpolate the short-term target output
Y_output = interp1(t, Yfinal, t_output); % note: we're adding back the column mean
Toutput = array2table(Y_output);

% generate variable names based on the number of transects
var_names = arrayfun(@(x) ['T', num2str(x)], 1:Ntr, 'UniformOutput', false);
Toutput.Properties.VariableNames = var_names;

Toutput = [table(datestr(t_output, 'yyyy-mm-dd'), 'VariableNames', {'Datetime'}) Toutput];
writetable(Toutput, [output_dir, loc, '_' run_id '_shoreline_prediction.csv']);

% Write each component of the shoreline projection to CSV files
components = {'longshore', 'crossshore', 'slr', 'trend'};
component_vars = {YLST, YST, YBRU, YVLT};
for k = 1:length(components)
    Tcomponent = array2table(component_vars{k});
    Tcomponent.Properties.VariableNames = var_names;
    Tcomponent = [table(datestr(t_output, 'yyyy-mm-dd'), 'VariableNames', {'Datetime'}) Tcomponent];
    writetable(Tcomponent, [output_dir, loc, '_' run_id '_comp_', components{k}, '.csv']);
end
