% ShoreShop 2.0 CoSMoS-COAST-CONV (convolution) model by Sean Vitousek.
% Edited by Jakob C. Christiaanse for sea turle beach modeling.
clear all; close all; clc;

% If the input data is not in the same directory as this file, add the path to the data
% cwd = 'YOUR/PATH/';   % add your path
cwd = '';               % leave empty if the data is in the same directory as this file

loc = 'AI_LB';          % specify the location identifier
slr_scen = 'hind'       % specify SLR scenario ('hind' for hindcast)

% specify run ID and output path
% you might need to adjust this to your directory structure
if strcmp(slr_scen, 'hind')
    run_id = slr_scen;
    output_dir = [cwd, loc, '/1_hindcast/'];
else
    run_id = ['slr' slr_scen];
    output_dir = [cwd, loc, '/2_forecast/'];
end

% Create output directory if it does not exist
if ~exist(output_dir, 'dir')
    mkdir(output_dir);
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Load shoreline data
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% load shorelines
fprintf('loading shoreline data ... '); % display progress
T = readtable([cwd, loc, '/0_input/', loc, '_shorelines.csv'], 'Delimiter', ','); % read in the shoreline observations to a matlab table
fprintf('done.\n'); % display progress

t_obs=round(datenum(datetime(T.Time,'InputFormat','yyyy-MM-dd HH:mm:ss+00:00')));         % get the time part of the table
Yraw=table2array(T(:,2:end));      % arrange all shoreline transects into a 2-D array (dimension 1 = time, dimension 2 = transect #)

% Detrend
% Y0=nanmean(Yraw);                  % calculate the mean value of each transect
% Y=Yraw-Y0;                         % detrend each shoreline time series (i.e., detrend each column) by removing the mean
Y=Yraw;                          % if Yraw is already detrended
Y0=[0, 0, 0];

Ntr=size(Y,2);                     % get the number of transects

% smooth the shorelines for each transect in time only (no spatial smoothing)
smooth_fac = 30;
Ysmooth = NaN(size(Y));
for j = 1:Ntr
    Ysmooth(:,j) = smoothn(Y(:,j), smooth_fac, 'robust');
end

% Identify and remove NaN rows (NaN over all transects)
Ymean=nanmean(Yraw,2);
id=isnan(Ymean);
t_obs(id)=[];
Y(id,:)=[];
Ysmooth(id,:)=[];

% figure; 
% subplot(2,1,1);plot(t_obs,Y,'.'); datetick('x')
% subplot(2,1,2);plot(t_obs,Ysmooth); datetick('x')

figure;
subplot(3,1,1);plot(t_obs,Y(:,1),'.',t_obs,Ysmooth(:,1)); datetick('x');
subplot(3,1,2);plot(t_obs,Y(:,2),'.',t_obs,Ysmooth(:,2)); datetick('x');
subplot(3,1,3);plot(t_obs,Y(:,3),'.',t_obs,Ysmooth(:,3)); datetick('x');
saveas(gcf, [output_dir, loc, '_smoothed_shorelines.jpg']);

% figure; plot(t_obs,Y,'.',t_obs,Ysmooth,'-');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Load forcing conditions
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

fprintf('loading wave/sea-level data ... '); % display progress

% load historical wave conditions
T=readtable([cwd, loc, '/0_input/', loc, '_waves_1979_2024.csv'],'Delimiter',',');     % load the wave height file to a matlab table
t_w=datenum(T.Time); % get the time vector
Hs=T.Hs;             % get Hs
Tp=T.Tp;             % get Tp
Dir=T.Dir;           % get Dir

% For future projections, we load projected waves and concat them to the hindcast
if ~strcmp(slr_scen, 'hind')
    T=readtable([cwd, loc, '/0_input/', loc, '_waves_2025_2100.csv'],'Delimiter',',');     % load the wave height file to a matlab table
    t_w=cat(1,t_w,datenum(T.Time)); % get the time vector
    Hs=cat(1,Hs,T.Hs);              % get Hs
    Tp=cat(1,Tp,T.Tp);              % get Tp
    Dir=cat(1,Dir,T.Dir);           % get Dir
end

% smooth waves (this is mostly done to remove the few odd NaN's that are present)
smooth_fac=0.001;                         % smoothing factor parameter
Hs_smooth=smoothn(Hs,smooth_fac); % also smooth the alongshore median wave height
Tp_smooth=smoothn(Tp,smooth_fac); % also smooth the alongshore median wave period

Dir0=nanmedian(Dir);                            % calculate the median wave direction (one scalar value)

Dir_rel=Dir-Dir0;                                % recenter the direction data by removing the median
Dir_rel(Dir_rel>180)=Dir_rel(Dir_rel>180)-360;   % recenter values greater than 180 degrees

Dir_rel(abs(Dir_rel)>45)=NaN;                    % get rid of highly oblique waves

Dir_smooth=smoothn(Dir_rel,smooth_fac); % smooth the direction

Dir_smooth=Dir_smooth-nanmedian(Dir_smooth);  % make sure the Dir_med has a zero median value

% check the wave direction
% figure; 
% subplot(3,1,1); plot(t_w,Hs_smooth,'b'); datetick('x')
% subplot(3,1,2); plot(t_w,Tp_smooth,'b'); datetick('x')
% subplot(3,1,3); plot(t_w,Dir_smooth,'b'); datetick('x')

% load the sea-level data
T=readtable([cwd, loc, '/0_input/', loc, '_sealevel_1979_2024.csv'],'Delimiter',',');     
t_sl=datenum(string(T.Time),'YYYY');                                        % get the time vector
SL=T.wl;

% if forecast, load projected sea-level conditions and concat them to the hindcasted ones
if ~strcmp(slr_scen, 'hind')
    T=readtable([cwd, loc, '/0_input/', loc, '_sealevel_2025_2100.csv'],'Delimiter',',');
    t_sl=cat(1,t_sl,datenum(string(T.Time),'YYYY')); % get the time vector
    slr = table2array(T(:, ['wl_' slr_scen]));
    SL=cat(1,SL,slr);
end

SL_smooth=smoothn(SL,1);                                                % smooth the sea-level curve
SL_w=interp1(t_sl,SL_smooth,t_w,'linear','extrap');             % interpolate SL onto the wave times
SL_w(t_w<t_sl(1))=SL_smooth(1);                               % use a flat part of the initial curve

fprintf('done.\n'); % display progress

% figure; plot(t_w,SL_w); datetick('x');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% model time stepping conditions
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

dt=1; % model time step in days 
t0=datenum(1979,1,01);    % model start time
if strcmp(slr_scen, 'hind')
    tstop=datenum(2024,12,31);   % model stop time
else
    tstop=datenum(2100,12,31);   % model stop time
end

% model time vector
t=(t0:dt:tstop)';

% number of time steps
len=length(t);

t_output=t';  % make time array for the short-term prediction

% get rid of obserations outside of model window
id=(t_obs<t0 | t_obs>tstop);
t_obs(id)=[];
Y(id,:)=[];
Ysmooth(id,:)=[];

% get forcing onto model time
[Hs]=interp1(t_w,Hs_smooth,t);
[Tp]=interp1(t_w,Tp_smooth,t);
[Dir]=interp1(t_w,Dir_smooth,t);
[SL]=interp1(t_w,SL_w,t,'linear','extrap');

% NOTE: we apply only ONE time series of wave height, period (not used),
% direction, and sea level to ALL transects.  Which is a bit different than
% applying individual/unique wave/SLR time series for each model transect.
% But Beach X is so small that unique time series are unnecessary any
% longshore variability will manifest in the alongshore variability of the
% model parameters... Besides unique wave time series are generally TOO
% FAR offshore to adequately capture the correct (i.e., 'nearshore') nature
% of the wave conditions, anyway.  

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% model parameters

% Yates (2009) model parameters (written in terms of the modification in my 2021 paper)

% (semi-optimized parameter values)
DT_cs=60;                       % equlibirum time scale parameter [days]
DY_cs=15;                       % shoreline excursion parameter [m]

% longshore transport parameter values
DT_ls=60;                       % equlibirum time scale parameter [days]
DY_ls=0;   % the longshore transport parameter (DIFFERENT FROM THE TYPICAL K in the normal COSMOS-COAST version ... can be positive or negative, for example)

% the equlibirum profile model (a.k.a. Bruun rule)
c=1;            % the Bruun coefficient [dimensionless scaling factor]
% tanAlpha = 1/50; % the bruun transgression slope (comment when using transect-specific slopes)

% Load transect-specific slopes
fprintf('loading transect specific beach slopes ... '); % display progress
slopes_file = ['slopes/', loc, '_slopes.csv'];
Tslopes = readtable(slopes_file);
tanAlpha = table2array(Tslopes(1,2:end)); % 1xNtr vector, skip first column
fprintf('done.\n')

% the long-term residual trend
vlt=0; % a linear trend [m/yr]

% calculate the long-term erosion rate in m/yr
vlt_max=NaN(1,Ntr);
for i=1:Ntr                                                         % for all transect
    id=~isnan(Y(:,i));                                              % get indices that are not NaN's
    vlt_max(1,i)=getfield(polyfit(t_obs(id),Y(id,i),1),{1})*365.25; % calculate the linear shoreline change rate
end

% the initial model parameter state vector (SEVEN parameters per transect)
%p= DT      DY      DT_ls  DY_ls    c     vlt      Y0 (i.e., the unknown initial condition)
p0=[DT_cs   DY_cs   DT_ls  DY_ls    c      0       0  ];  % the initial guess of the model parameters
lb=[20      2       20     -5000    0.5  -1    -70 ];  % ... and its lower and 
ub=[300     100     365    5000    1.5   1     70  ];  % ... upper bounds

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% THE MODEL PHYSICS (which is surprisingly short, IMO)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% NOTE THAT THE INPUT VARIABLE 'p' (used in the anonymous functions below) 
% is the state vector that contains the optimization parameters for a given transect ... 
% we optimize the model by determining the best 'p'

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% (wave-driven) cross-shore model (i.e., the modified Yates model)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% the wave energy anomaly scaled by parameters DY (i.e., p(2)) and DT (i.e., p(1)).
f_cs=@(p,t,Hs) -p(2)/p(1)*(detrend(Hs.^2));  % the modified Yates convolution forcing function (written as a matlab anonymous function)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% longshore transport model (i.e., a CERC-like model)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% the longshore flux function.  Note that p(3) is longshore excursion parameter DY_ls, which varies between transects
Q   =@(p,t,Hs,alpha) p(4).*sind(2*alpha);   % the (CERC-like) longshore sediment transport flux (written as a matlab anonymous function)

% the longshore forcing function
f_ls=@(p,t,Hs,alpha) 1./p(3)*detrend(Q(p,t,Hs,alpha),0);   % the detrended lonshore sediment flux, scaled by the (inverse of the) time scale DT (i.e., p(1))

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% (sea-level-driven) cross-shore model (i.e., the Bruun rule)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% the Bruun rule (comment when using transect-specific slopes, will be defined within the transect loop)
% Ybru=@(p,t,S) -p(5)/tanAlpha*(S-S(1));  % the Bruun model (written as a matlab anonymous function) 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% the residual trend term
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% a linear residual trend
Yvlt=@(p,t) p(6)/365.25*(t-t(1))+p(7);         % the residual trend (written as a matlab anonymous function)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% the combined cross-shore + longshore transport model, written as a convolution
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

f=@(p,t,Hs,alpha) f_cs(p,t,Hs)+f_ls(p,t,Hs,alpha);       % the convolution focring function (written as a matlab anonymous function) 
g_cs=@(p,t)          (1-dt/p(1)).^(t/dt).*(t>=0);        % the convolution  memory function (written as a matlab anonymous function), which is related to the time scale parameter DT (i.e., p(1))   
g_ls=@(p,t)          (1-dt/p(3)).^(t/dt).*(t>=0);        % the convolution  memory function (written as a matlab anonymous function), which is related to the time scale parameter DT (i.e., p(1))   

% the cross-shore shoreline change component only
Yst=@(p,t,Hs) cat(1,0,dt*getfield(fconv2(f_cs(p,t,Hs),g_cs(p,t-t(1))),{1:length(t)-1,1}));

% the longshore shoreline change component only
Ylst=@(p,t,Hs,alpha) cat(1,0,dt*getfield(fconv2(f_ls(p,t,Hs,alpha),g_ls(p,t-t(1))),{1:length(t)-1,1}));

% the combined cross-shore + longshore equlibirum + SLR + trend model, written as a convolution
% Comment this part out if you want to use transect-specific slopes, as the model will be defined within the transect loop
% Ymod=@(p,t,Hs,alpha,S) Yst(p,t,Hs)+Ylst(p,t,Hs,alpha)+Ybru(p,t,S)+Yvlt(p,t);  % the full model, written as a convolution (note: a convolution along each column, where each column represents a different transect)

% Note that while written this way, the model does not do ANY traditional
% time-stepping. Instead, time series of wave conditions are used to
% DIRECTLY calculate the shoreline time series, rather than
% updating/time-stepping Y^(N+1) from Y^N+f^N.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% THE MODEL TO CALIBRATE/OPTIMIZE
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% the observed shoreline data set that is used for model calibration
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

t_obs_cal=t_obs;   % the observation times used for model calibration
Yobs=Ysmooth;      % the (smoothed) observations used for model calibration (CALIBRATE USING SMOOTHED DATA)

% t_obs_cal=ti_obs;   % the (daily interpolated) observation times used for model calibration
% Yobs=Yi;            % the (daily interpolated) observations used for calibration

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% the model to optimize (as a function of the model parameters p)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Comment this part out if you want to use transect-specific slopes, as the model will be defined within the transect loop
% the vector of model output for the calibration scenario.  Note that the only input is p, the parameters.  The input conditions (t,Hs,Dir) define the calibration portion/period
% Ycal=@(p)  Ymod(p,t,Hs,Dir,SL);
    
% the model interpolated onto the observations (to evaluate the skill metrics)
% Ycali=@(p) interp1(t,Ycal(p),t_obs_cal);    % the vector of model output interpolated to the observation points

% skill metric arrays (columns represent each transect)
P=NaN(length(p0),Ntr);
RMSE_opt=NaN(1,Ntr);
RMSEn_opt=NaN(1,Ntr);
STDn_opt=NaN(1,Ntr);
COR_opt=NaN(1,Ntr);
IDX_opt=NaN(1,Ntr);
LAMBDA_opt=NaN(1,Ntr);
L_opt=NaN(1,Ntr);

% save the individual shoreline components for all transects
Yfinal=NaN(length(t),Ntr);
YLST=NaN(length(t),Ntr);
YST=NaN(length(t),Ntr);
YBRU=NaN(length(t),Ntr);
YVLT=NaN(length(t),Ntr);

% start with p=p0, then use the optimized version of the previous transect for the initial condition for the next transect iteration
%p=p0; % initial parameter state vector is from the initial conditions (Although I'm not sure how this actually works in parallel, so I'm not including it)

%% Note: The following 'for'-loop can be run in parallel using 'parfor' instead of the typical (serial) 'for' loop. If you want to run it in parallel, make sure you have the Parallel Computing Toolbox installed and uncomment the three lines below.
% parpool(Ntr); % set up a parallel pool w/ one worker for each transect.
% fprintf('Running optimization in parallel (this may take a few minutes) ... \n'); % display progress
% parfor i=1:Ntr  % FOR EACH TRANSECT ... optimize for each transects individually

% Serial for-loop (not parallel, see above)
for i=1:Ntr  % FOR EACH TRANSECT ... optimize for each transects individually

    STDt=nanstd(Yobs(:,i));        % the standard deviation of the observations used later to calculate the skill metrics

    % Define the bruun and full model functions for each transect specific slope (comment this part out if you want to use a global slope, and uncomment the individual function definitions above)
    Ybru = @(p, t, S) -p(5)/tanAlpha(i)*(S-S(1));
    Ymod = @(p, t, Hs, alpha, S) Yst(p, t, Hs) + Ylst(p, t, Hs, alpha) + Ybru(p, t, S) + Yvlt(p, t);
    Ycal = @(p) Ymod(p, t, Hs, Dir, SL);
    Ycali = @(p) interp1(t, Ycal(p), t_obs_cal);

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % skill metrics (as a function of the model parameters p - which we seek to optimize)
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    ERROR=@(p) Ycali(p)-Yobs(:,i);                      % the difference between model and observations
    STD  =@(p) nanstd(Ycali(p));                        % the standard deviation of the model
    STDn =@(p) STD(p)/STDt;                             % the normalized standard deviation
    MSE  =@(p) nanmean((ERROR(p)).^2);                  % mean square error (MSE)
    RMSE =@(p) sqrt(mean(ERROR(p).^2));                 % the root-mean-square error (RMSE)
    RMSEn=@(p) RMSE(p)/STDt;                            % the normalized RMSE
    COR  =@(p) diag(corrcoef(Ycali(p),Yobs(:,i)),1);    % the correlation coefficient    
    
    IDX_AGREEMENT=@(p) 1-sum((Yobs(:,i)-Ycali(p)).^2)/sum( (abs(Ycali(p)-nanmean(Yobs(:,i)))+abs(Yobs(:,i)-nanmean(Yobs(:,i)))).^2 ); % index of agreement
    LAMBDA       =@(p) 1-MSE(p)/(var(Yobs(:,i))+var(Ycali(p))+(nanmean(Yobs(:,i))-nanmean(Ycali(p))).^2);                             % lambda

    % the loss function to optimize (based on the target given in the ShoreShop2.0 guidelines)
    L=@(p) sqrt((0-RMSEn(p))^2+(1-COR(p))^2+(1-STDn(p))^2); % the loss function to be minimized

    % optimization options
    options = optimset('Display','off'); % turn 'off' all the optimization dispay progress, you can also replace 'off' with 'iter' if you want the progress to display

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % OPTIMIZE
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    fprintf('running optimization for transect %d (this may take a few minutes) ... ',i); % display progress
    tic          % start timer
    p = fmincon(L,p0,[],[],[],[],lb,ub,[],options); % minimize the loss function subject to the lower and upper bounds (lb and ub, respectively, given above)
    twall=toc;   % stop timer

    % display wall-clock time
    if twall>60
        fprintf('done. wall-clock time = %.1f minutes. ',twall/60); % print total time in minutes
    else
        fprintf('done. wall-clock time = %.1f seconds. ',twall);    % or print total time in seconds
    end

    % save optimized parameters for all transects
    P(:,i)=p;

    % save output the final optimized skill metrics
    RMSE_opt(i)=RMSE(p);
    RMSEn_opt(i)=RMSEn(p);
    STDn_opt(i)=STDn(p);
    COR_opt(i)=COR(p);
    IDX_opt(i)=IDX_AGREEMENT(p);
    LAMBDA_opt(i)=LAMBDA(p);
    L_opt(i)=L(p);

    % report RMS error for each transect after calibration
    fprintf('(optimized) model RMSE = %.2f m.\n',RMSE_opt(i));

    % save output the final shoreline prediction
    Yfinal(:,i)=Ycal(p);

    % save output the final shoreline prediction
    YLST(:,i)=Ylst(p,t,Hs,Dir);
    YST(:,i)= Yst(p,t,Hs    );
    YBRU(:,i)=Ybru(p,t,SL    );
    YVLT(:,i)=Yvlt(p,t       );

    % make an example figure after calibration
    % figure; set(gcf,'Position',[200+(i-1)*20 400-(i-1)*20 1400 600]); hold on; box on;
    % han1=plot([t_obs(2)-100 t_obs(end)+500],[0 0],'k--'); set(han1,'HandleVisibility','off')
    % han1=plot(t,Yfinal(:,i),'-r',t_obs_cal,Yobs(:,i),'bo'); set(han1,'MarkerSize',6,'MarkerFaceColor','b','MarkerEdgeColor','k');
    % %han1=plot(t,YLST(:,i),'-r',t,YST(:,i),'-m',t,YBRU(:,i),'-c',t,YVLT(:,i),'-g'); datetick('x','keeplimits'); set(han1,'LineWidth',1.5); axis tight;
    % %han1=plot(t_obs_cal(1),Yobs(1,i),'co'); set(han1,'MarkerSize',10,'MarkerFaceColor','c','MarkerEdgeColor','k');
    % set(gca,'XLim',[t_obs(2)-100 t_obs(end)+500],'FontSize',16); datetick('x','keeplimits');
    % legend('model','(smoothed) observations','Location','SouthEast');
    % ylabel('shoreline position [m]');
    % title(['transect #',num2str(i)]);
    % drawnow;

end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% make .csv output
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% make/interpolate the short-term target output
Y_output = interp1(t, Yfinal, t_output) + Y0; % note: we're adding back the column mean
Toutput = array2table(Y_output);

% generate variable names based on the number of transects
var_names = arrayfun(@(x) ['T', num2str(x)], 1:Ntr, 'UniformOutput', false);
Toutput.Properties.VariableNames = var_names;

Toutput = [table(datestr(t_output, 'yyyy-mm-dd'), 'VariableNames', {'Datetime'}) Toutput];
writetable(Toutput, [output_dir, loc, '_' run_id '_shoreline_prediction.csv']);

% Write the smoothed shorelines to a CSV file with timestamps
if strcmp(slr_scen, 'hind')
    Tsmooth = array2table(Ysmooth + Y0);
    Tsmooth.Properties.VariableNames = var_names;
    Tsmooth = [table(datestr(t_obs, 'yyyy-mm-dd'), 'VariableNames', {'Datetime'}) Tsmooth];
    writetable(Tsmooth, [output_dir, loc, '_shoreline_smoothed.csv']);
end

% Write the parameters to a CSV file with parameter names
param_names = {'DT_cs', 'DY_cs', 'DT_ls', 'DY_ls', 'c', 'vlt', 'Y0'};
Tparameters = array2table([param_names', num2cell(P)], 'VariableNames', ['Parameter', var_names]);
writetable(Tparameters, [output_dir, loc, '_' run_id '_parameters.csv']);

% Write the skill metrics to a separate CSV file
skill_names = {'RMSE', 'RMSEn', 'STDn', 'COR', 'IDX', 'LAMBDA', 'L'};
skill_matrix = [RMSE_opt; RMSEn_opt; STDn_opt; COR_opt; IDX_opt; LAMBDA_opt; L_opt];
Tskills = array2table([skill_names', num2cell(skill_matrix)], 'VariableNames', ['Metric', var_names]);
writetable(Tskills, [output_dir, loc, '_' run_id '_skill_metrics.csv']);

% Write each component of the shoreline projection to CSV files
components = {'longshore', 'crossshore', 'slr', 'trend'};
component_vars = {YLST, YST, YBRU, YVLT};
for k = 1:length(components)
    Tcomponent = array2table(component_vars{k} + Y0);
    Tcomponent.Properties.VariableNames = var_names;
    Tcomponent = [table(datestr(t_output, 'yyyy-mm-dd'), 'VariableNames', {'Datetime'}) Tcomponent];
    writetable(Tcomponent, [output_dir, loc, '_' run_id '_comp_', components{k}, '.csv']);
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Make an output plot
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
xp0=0.09;
yp0=0.74;
width=0.84;
height=0.22;
xsep=0.05;
ysep=0.012;

FONTSIZE=14;

left_color = [0 0 1];
right_color = [255 0 0]/255;
set(gcf,'defaultAxesColorOrder',[left_color; right_color]);

figure; set(gcf,'Position',[100 100 1000 1000],'PaperPositionMode','auto','color','w');
nsubplot=1; ax=subplot('position',[xp0 yp0-(nsubplot-1)*(height+ysep)+0.03 width height]);  hold on; box on;
yyaxis left
han1=plot(t,Hs,'b'); axis tight; axis tight; datetick('x','keepticks','keeplimits'); ylabel('H_s [m]'); set(han1,'color',left_color);
set(gca,'XTick',datenum(1980:10:2100,1,1),'FontSize',14,'XTickLabel',[]); datetick('x','keepticks','keeplimits'); set(gca,'XTickLabel',[]);

yyaxis right
han1=plot(t,SL,'r'); axis tight; axis tight; datetick('x','keepticks','keeplimits'); ylabel('Sea level [m]');
set(han1,'color',right_color,'LineWidth',2);
axis([t0 tstop -0.35 0.8]);

for i=1:Ntr
    nsubplot=i+1; ax=subplot('position',[xp0 yp0-(nsubplot-1)*(height+ysep) width height]);  hold on; box on;
    plot(t_obs_cal,Yobs(:,i)+Y0(i),'b.',t,Yfinal(:,i)+Y0(i),'r'); axis tight; datetick('x','keepticks','keeplimits');
    ylabel({['transect #',num2str(i)],'shoreline position [m]'})
    set(gca,'XTick',datenum(1980:10:2100,1,1),'FontSize',14); datetick('x','keepticks','keeplimits');
    if i<Ntr
        set(gca,'XTickLabel',[]);
    end
end
legend('observations','model','Position',[0.763666666666667 0.436854830775502 0.146333333333333 0.0471666666666665]);

saveas(gcf,[output_dir, loc, '_' run_id '_shoreline_prediction.jpg']);

% shut down parallel pool (in case the for loop was run in parallel
delete(gcp('nocreate'))

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% End of the model